<?php

namespace Espo\Custom\Hooks\Campaign;

use Espo\Core\Exceptions\BadRequest;
use Espo\Core\Mail\EmailFactory;
use Espo\Core\Mail\EmailSender;
use Espo\Core\Utils\Config;
use Espo\ORM\Entity;
use Espo\ORM\EntityManager;

use Espo\Core\Job\JobSchedulerFactory;
use Espo\Core\Job\QueueName;

class AfterSave
{
    public function __construct(
        private EmailSender   $emailSender,
        private EmailFactory  $emailFactory,
        private Config        $config,
        private EntityManager $entityManager,
        JobSchedulerFactory $jobSchedulerFactory
    ){
        $this->jobSchedulerFactory = $jobSchedulerFactory;
    }

    /**
     * Regras de transição de status antes de salvar.
     */
    public function beforeSave(Entity $entity, array $options): void
    {
        $log = $GLOBALS['log'] ?? null;
        $log?->info('Init beforeSave.');

        // Só valida quando o status mudou.
        if (!$entity->isAttributeChanged('cNivel')) {
            return;
        }
    }

    public function afterSave(Entity $entity, array $options): void
    {
        $log = $GLOBALS['log'] ?? null;
        $log?->info('Init afterSave - Campaign.');

        /**
         * Evita loop infinito em updates internos
         */
        if (!empty($options['isInternalUpdate'])) {
            return;
        }

        /**
         * 2) Buscar o cIdAcao recarregando do banco
         */
        $idAcao = $entity->get('cIdAcao');
        $budget = $entity->get('budget');

        if (!$idAcao) {
            // Recarrega a entidade direto do banco
            $reloaded = $this->entityManager->getEntity('Campaign', $entity->get('id'));

            if ($reloaded) {
                $idAcao = $reloaded->get('cIdAcao');
                $log?->info('cIdAcao recarregado do banco: ' . var_export($idAcao, true));
            } else {
                $log?->warning('Não foi possível recarregar Campaign no afterSave.');
            }
        }

        if ($idAcao) {
            $nomeFinal = 'Ação - ' . $idAcao;

            // Usa SEMPRE o objeto recarregado para garantir consistência
            $targetEntity = isset($reloaded) && $reloaded ? $reloaded : $entity;

            if ($targetEntity->get('name') !== $nomeFinal) {
                $log?->info('Renomeando campanha para: ' . $nomeFinal);

                $targetEntity->set('name', $nomeFinal);
                $targetEntity->set('cCInvestimentoNecessario', $budget);

                // Segundo save interno, sem recursão
                $this->entityManager->saveEntity($targetEntity, [
                    'isInternalUpdate' => true
                ]);
            }
        } else {
            $log?->info('cIdAcao ainda nulo mesmo após reload.');
        }

        /**
         * Envio de notificação para o usuario no caso de uma mudança de status da ação. 
         * TODO: Definir para quem vai ser enviado
         */
        if ($entity->isAttributeChanged('cNivel')) {
            $status = $entity->get('cNivel');
            $description = $entity->get('description');
            $log?->info('Mudou de status: '.$status. ' - Briefing: '.$description);

            $assignedUserId = $entity->get('assignedUserId');
            if ($assignedUserId) {
                $user = $this->entityManager->getEntity('User', $assignedUserId);
                if ($user) {
                    $userEmail = $user->get('emailAddress');
                    $log?->info('Email do usuário: ' . $userEmail);
                    $this->sendEmail($userEmail, $entity);
                }
            } else {
                $log?->info('Campanha sem usuário designado.');
            }
        }
    }

    /**
     * Envio de e-mail simples (texto) notificando a mudança de status.
     * ralq dpab aawd wcvr
     */
    public function sendEmail(string $emailTo, Entity $campaignEntity): void
    {
        $log = $GLOBALS['log'] ?? null;

        $campaignName = (string) $campaignEntity->get('name');
        $newStatus    = (string) $campaignEntity->get('cNivel');
        $campaignId   = $campaignEntity->getId();

        $siteUrl      = (string) $this->config->get('siteUrl');
        $campaignLink = rtrim($siteUrl, '/') . '/#Campaign/view/' . $campaignId;

        $subject = "Atualização de Status";        
        
        $html_body = "
        <p>
            A <strong>{$campaignName}</strong> teve seu status atualizado para:<br>
            <strong>{$newStatus}</strong>
        </p>

        <p>
            Para visualizar os detalhes, acesse:<br>
            <a href='{$campaignLink}'>{$campaignLink}</a>
        </p>

        <p>
            Atenciosamente,<br>
            <strong>Suporte Soft CRM</strong>
        </p>
    ";

        //$email = $this->emailFactory->create();
        //$email->setSubject($subject);
        //$email->setBody($html_body);
        //$email->addToAddress($emailTo);
        //$email->setIsPlain();
        //$email->setIsHtml();

        try {
            //$this->emailSender->send($email);
            
        $this->jobSchedulerFactory->create()
                ->setClassName('Espo\\Custom\\Jobs\\SendMentionEmail') // Nome completo da classe do job
                ->setQueue(QueueName::E0) // Queue 'e0' é otimizada para envios de email (processa o mais rápido possível, um por um)
                ->setData([
                    'emailTo' => $emailTo,
                    'htmlBody' => $html_body,
                    'subject' => $subject
                ])
                ->schedule();    

            $log?->info("Email enviado com sucesso para: {$emailTo} (campanha: {$campaignName})");
        } catch (\Throwable $e) {
            $log?->error('Erro ao enviar email: ' . $e->getMessage());
        }
    }
}
